<?php

namespace admin\models;

use yii\base\Model;
use yii\data\ActiveDataProvider;
use common\models\Order;
use common\components\DateComponent;
use common\components\CurrencyComponent;

/**
 * OrderSearch represents the model behind the search form of `common\models\Order`.
 */
class OrderSearch extends Order
{
    public $member_fullname, $start_date, $end_date, $sipto, $voucher_code;
    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id', 'member_id', 'order_recipient_province_id', 'order_recipient_city_id', 'created_by', 'updated_by', 'is_deleted'], 'integer'],
            [['order_code', 'order_date', 'order_recipient_fullname', 'order_recipient_address', 'order_recipient_province_name', 'order_recipient_city_name', 'order_recipient_phone', 'order_recipient_email', 'order_payment_method', 'order_status', 'created_on', 'updated_on', 'member_fullname', 'start_date', 'end_date', 'sipto', 'order_note', 'voucher_code'], 'safe'],
            [['order_subtotal', 'order_shipping_fee', 'order_grand_total'], 'number'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function scenarios()
    {
        // bypass scenarios() implementation in the parent class
        return Model::scenarios();
    }

    /**
     * Creates data provider instance with search query applied
     *
     * @param array $params
     *
     * @return ActiveDataProvider
     */
    public function search($params, $page = true)
    {
        $query = Order::find();
        $query->joinWith(['member AS member', 'voucher AS voucher']);

        // add conditions that should always apply here

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => $page ? [
                'pageSize' => (new \common\models\Setting)->getSettingValueByName('record_per_page')
            ] : false,
            'sort'=> [
                'attributes' => [
                    'order_code',
                    'confirmation_date',
                    'order_date',
                    'order_recipient_fullname',
                    'order_recipient_phone',
                    'order_recipient_email',
                    'order_subtotal',
                    'order_shipping_fee',
                    'order_grand_total',
                    'order_status',
                    'order_payment_method',
                    'order_note',
                    'member_fullname' => [
                        'asc' => ['member.member_fullname' => SORT_ASC],
                        'desc' => ['member.member_fullname' => SORT_DESC],
                    ],
                    'voucher_code' => [
                        'asc' => ['voucher.voucher_code' => SORT_ASC],
                        'desc' => ['voucher.voucher_code' => SORT_DESC],
                    ],
                    'sipto' => [
                        'asc' => ['order_recipient_fullname' => SORT_ASC],
                        'desc' => ['order_recipient_fullname' => SORT_DESC],
                    ],
                ],
                'defaultOrder' => ['order_date' => SORT_DESC],
            ],
        ]);

        $this->load($params);

        if (!$this->validate()) {
            // uncomment the following line if you do not want to return any records when validation fails
            // $query->where('0=1');
            return $dataProvider;
        }

        // grid filtering conditions
        $query->andFilterWhere([
            'id' => $this->id,
            'member_id' => $this->member_id,
            'order_date' => $this->order_date,
            'order_recipient_province_id' => $this->order_recipient_province_id,
            'order_recipient_city_id' => $this->order_recipient_city_id,
            'order_subtotal' => $this->order_subtotal,
            'order_shipping_fee' => $this->order_shipping_fee,
            'order_grand_total' => $this->order_grand_total,
            'created_by' => $this->created_by,
            'updated_by' => $this->updated_by,
            'created_on' => $this->created_on,
            'updated_on' => $this->updated_on,
            'tbl_order.is_deleted' => 0,
        ]);

        $query->andFilterWhere(['like', 'order_code', $this->order_code])
            ->andFilterWhere(['like', 'order_recipient_fullname', $this->order_recipient_fullname])
            ->andFilterWhere(['like', 'order_recipient_address', $this->order_recipient_address])
            ->andFilterWhere(['like', 'order_recipient_province_name', $this->order_recipient_province_name])
            ->andFilterWhere(['like', 'order_recipient_city_name', $this->order_recipient_city_name])
            ->andFilterWhere(['like', 'order_recipient_phone', $this->order_recipient_phone])
            ->andFilterWhere(['like', 'order_recipient_email', $this->order_recipient_email])
            ->andFilterWhere(['like', 'order_payment_method', $this->order_payment_method])
            ->andFilterWhere(['like', 'order_status', $this->order_status])
            ->andFilterWhere(['like', 'order_note', $this->order_note])
            ->andFilterWhere(['like', 'voucher.voucher_code', $this->voucher_code])
            ->andFilterWhere(['like', 'member.member_fullname', $this->member_fullname]);

        if ($this->start_date != "" && preg_match('/^(\d{2})-(\d{2})-(\d{4})$/', $this->start_date))
            $query->andWhere('DATE(order_date) >= "'.date("Y-m-d", strtotime($this->start_date)).'"');
        if ($this->end_date != "" && preg_match('/^(\d{2})-(\d{2})-(\d{4})$/', $this->end_date))
            $query->andWhere('DATE(order_date) <= "'.date("Y-m-d", strtotime($this->end_date)).'"');

        if(!empty($this->sipto)){
            $query->andWhere("tbl_order.id IN (select id from tbl_order where CONCAT(COALESCE(order_recipient_fullname, ''), ' ', COALESCE(order_recipient_address, ''), ' ',  COALESCE(order_recipient_province_name, ''), ' ',  COALESCE(order_recipient_city_name, '')) LIKE '%".$this->sipto."%' )", []);
        }

        return $dataProvider;
    }

    public static function getTotal($provider, $fieldName)
    {
        $total = 0;

        foreach ($provider as $item) {
            $total += $item[$fieldName];
        }

        return $total;
    }

    public static function getExcel($provider, $arr = [])
    {
        date_default_timezone_set('Asia/Jakarta');
        
        $firstCol = "A";
        $endCol = "N";
    
        $objPHPExcel = new \PHPExcel();
        $objPHPExcel->setActiveSheetIndex(0);
        $objPHPExcel->getActiveSheet()->mergeCells("A1:N1")->setCellValue('A1', 'Order Report')->getStyle('A1')->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_CENTER), 'font' => array('bold' => 'true', 'size' => '18')));

        $rowNum = 3;
        $isFilter = false;

        $date = '';
        if(!empty($arr['start_date']) && !empty($arr['end_date'])){
            $date = 'From '.$arr['start_date'].' to '.$arr['end_date'];
        }
        elseif(!empty($arr['start_date']) && empty($arr['end_date'])){
            $date = 'Since '.$arr['start_date'];
        }
        elseif(empty($arr['start_date']) && !empty($arr['end_date'])){
            $date = 'Until '.$arr['end_date'];
        }

        if(!empty($date)){
            $objPHPExcel->getActiveSheet()->setCellValue('A'.$rowNum, 'Date :');
            $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum, $date)
            ->getStyle('B'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_LEFT)));
            $rowNum++;
            $isFilter = true;
        }
        
        
        if(!empty($arr)){
            foreach ($arr as $key => $value) {
                if(!empty($value) && $key != 'start_date' && $key != 'end_date'){
                    $objPHPExcel->getActiveSheet()->setCellValue('A'.$rowNum, ((new Order)->getAttributeLabel($key) != 'sipto' ? (new Order)->getAttributeLabel($key) : 'Ship to').' :');
                    $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum, $value)
                    ->getStyle('B'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_LEFT)));
                    $rowNum++;
                    $isFilter = true;
                }
            }
        }

        if($isFilter){
            $rowNum++;
        }

        $objPHPExcel->getActiveSheet()->setCellValue('A'.$rowNum, 'No');
        $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum, 'Member Full Name');
        $objPHPExcel->getActiveSheet()->setCellValue('C'.$rowNum, 'Code');
        $objPHPExcel->getActiveSheet()->setCellValue('D'.$rowNum, 'Date');
        $objPHPExcel->getActiveSheet()->setCellValue('E'.$rowNum, 'Ship to');
        $objPHPExcel->getActiveSheet()->setCellValue('F'.$rowNum, 'Phone');
        $objPHPExcel->getActiveSheet()->setCellValue('G'.$rowNum, 'Email');
        $objPHPExcel->getActiveSheet()->setCellValue('H'.$rowNum, 'Subtotal');
        $objPHPExcel->getActiveSheet()->setCellValue('I'.$rowNum, 'Shipping fee');
        $objPHPExcel->getActiveSheet()->setCellValue('J'.$rowNum, 'Voucher code');
        $objPHPExcel->getActiveSheet()->setCellValue('K'.$rowNum, 'Grand Total');
        $objPHPExcel->getActiveSheet()->setCellValue('L'.$rowNum, 'Payment Method');
        $objPHPExcel->getActiveSheet()->setCellValue('M'.$rowNum, 'Status');
        $objPHPExcel->getActiveSheet()->setCellValue('N'.$rowNum, 'Note');

        $objPHPExcel->getActiveSheet()->getStyle('A'.$rowNum.':N'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_CENTER), 'font' => array('bold' => 'true'), 'fill' => array('type' => \PHPExcel_Style_Fill::FILL_SOLID, 'color' => array('rgb' => 'CCCCCC'))));
        $rowNum++;

        $subtotal = 0;
        $shippingfee = 0;
        $grandtotal = 0;
        foreach($provider as $i => $item) {
            $subtotal += $item->order_subtotal;
            $shippingfee += $item->order_shipping_fee;
            $grandtotal += $item->order_grand_total;

            $objPHPExcel->getActiveSheet()->setCellValue('A'.$rowNum,  $i+1)
            ->getStyle('A'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP, 'horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_CENTER)));
            $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum,  !empty($item->member->member_fullname) ? $item->member->member_fullname : '')
            ->getStyle('B'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);
            $objPHPExcel->getActiveSheet()->setCellValue('C'.$rowNum,  $item->order_code)
            ->getStyle('C'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('D'.$rowNum,  DateComponent::getHumanizedDate($item->order_date))
            ->getStyle('D'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));

            $sipto = str_replace('<br />', "", $item->sipto);
            $objPHPExcel->getActiveSheet()->setCellValue('E'.$rowNum, $sipto);
            // ->getStyle('E'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);
            $objPHPExcel->getActiveSheet()->getStyle('E'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);

            $objPHPExcel->getActiveSheet()->setCellValue('F'.$rowNum, $item->order_recipient_phone)
            ->getStyle('F'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('G'.$rowNum, $item->order_recipient_email)
            ->getStyle('G'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('H'.$rowNum, CurrencyComponent::formatMoney($item->order_subtotal))
            ->getStyle('H'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('I'.$rowNum, CurrencyComponent::formatMoney($item->order_shipping_fee))
            ->getStyle('I'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));

            $voucher = "";
            if(!empty($item->voucher->voucher_code)){
                $voucher = $item->voucher->voucher_code;
                if(!empty($item->voucher_value_rupiah)){
                    $voucher .= "\n".CurrencyComponent::formatMoney($item->voucher_value_rupiah);
                }
            }
            $objPHPExcel->getActiveSheet()->setCellValue('J'.$rowNum, $voucher)
            ->getStyle('J'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);

            $objPHPExcel->getActiveSheet()->setCellValue('K'.$rowNum, CurrencyComponent::formatMoney($item->order_grand_total))
            ->getStyle('K'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('L'.$rowNum, $item->getMethodLabels($item->order_payment_method))
            ->getStyle('L'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
            $objPHPExcel->getActiveSheet()->setCellValue('M'.$rowNum, $item->getStatusLabels($item->order_status))
            ->getStyle('M'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));

            $note = !empty($item->order_note) ? str_replace('<br />', "", $item->order_note) : '';
            $objPHPExcel->getActiveSheet()->setCellValue('N'.$rowNum, $note)
            ->getStyle('N'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);

            //item
            $rowNum++;
            $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum, 'Product');
            $objPHPExcel->getActiveSheet()->setCellValue('C'.$rowNum, 'Qty');
            $objPHPExcel->getActiveSheet()->setCellValue('D'.$rowNum, 'Price');
            $objPHPExcel->getActiveSheet()->setCellValue('E'.$rowNum, 'Subtotal');
            $objPHPExcel->getActiveSheet()->setCellValue('F'.$rowNum, 'Weight');

            $objPHPExcel->getActiveSheet()->getStyle('B'.$rowNum.':F'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_CENTER), 'font' => array('bold' => 'true'), 'fill' => array('type' => \PHPExcel_Style_Fill::FILL_SOLID, 'color' => array('rgb' => 'EEEEEE'))));
            $rowNum++;

            foreach ($item->orderDetails as $key => $value) {
                $objPHPExcel->getActiveSheet()->setCellValue('B'.$rowNum,  !empty($value->product_attribute_group_id) ? $value->product->product_name.' '.$value->getAttributeOrder() : $value->product->product_name)
                ->getStyle('B'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)))->getAlignment()->setWrapText(true);
                $objPHPExcel->getActiveSheet()->setCellValue('C'.$rowNum, $value->detail_qty )
                ->getStyle('C'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
                $objPHPExcel->getActiveSheet()->setCellValue('D'.$rowNum, CurrencyComponent::formatMoney($value->detail_price) )
                ->getStyle('D'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
                $objPHPExcel->getActiveSheet()->setCellValue('E'.$rowNum, CurrencyComponent::formatMoney($value->detail_subtotal) )
                ->getStyle('E'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT, 'vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));

                $Weight = $value->product->product_weight_in_gram * $value->detail_qty;
                $Weight = empty($Weight) ? 0 : ceil($Weight / 1000);
                $objPHPExcel->getActiveSheet()->setCellValue('F'.$rowNum, $Weight )
                ->getStyle('F'.$rowNum)->applyFromArray(array('alignment' => array('vertical' => \PHPExcel_Style_Alignment::VERTICAL_TOP)));
                $rowNum++;
            }
            $rowNum++;
        }

        $objPHPExcel->getActiveSheet()->setCellValue('H'.$rowNum, CurrencyComponent::formatMoney($subtotal));
        $objPHPExcel->getActiveSheet()->setCellValue('I'.$rowNum, CurrencyComponent::formatMoney($shippingfee));
        $objPHPExcel->getActiveSheet()->setCellValue('K'.$rowNum, CurrencyComponent::formatMoney($grandtotal));
        $objPHPExcel->getActiveSheet()->getStyle('H'.$rowNum.':K'.$rowNum)->applyFromArray(array('alignment' => array('horizontal' => \PHPExcel_Style_Alignment::HORIZONTAL_RIGHT), 'font' => array('bold' => 'true')));

        //auto fit the cell's width with the content
        foreach(range($firstCol, $endCol) as $columnID) {
            if ($columnID != "E" && $columnID != "B")
                $objPHPExcel->getActiveSheet()->getColumnDimension($columnID)->setAutoSize(true);
        }

        // set fixed width for address and member name
        $objPHPExcel->getActiveSheet()->getColumnDimension('B')->setWidth(50);
        $objPHPExcel->getActiveSheet()->getColumnDimension('E')->setWidth(50);

        //auto fit the cell's height with the content
        foreach($objPHPExcel->getActiveSheet()->getRowDimensions() as $rd) {
            $rd->setRowHeight(-1);
        }
        
        // Save a xls file
        $filename = "Order_report_".date("d-m-Y-His").".xls";
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename='.$filename .' ');
        header('Cache-Control: max-age=0');
        
        $objWriter = \PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
        $objWriter->save('php://output');
        
        die();
    }
}
