<?php
namespace frontend\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\helpers\Url;
use common\models\ProductCategory;
use common\models\ProductCategoryBrand;
use common\models\Slide;
use common\models\Article;
use common\models\Segmentation;
use common\models\IndoDistribution;
use common\models\InternationalDistribution;
use app\components\BaseController;
use frontend\components\StringComponent;

/**
 * Site controller
 */
class SiteController extends BaseController
{

    /**
     * {@inheritdoc}
     */
    public function actions()
    {
        return [
            'error' => [
                'class' => 'yii\web\ErrorAction',
            ],
            'captcha' => [
                'class' => 'yii\captcha\CaptchaAction',
                'fixedVerifyCode' => YII_ENV_TEST ? 'testme' : null,
            ],
        ];
    }

    /**
     * Displays homepage.
     *
     * @return mixed
     */
    public function actionIndex()
    {
        BaseController::$page_caption = 'Home';

        $categories = ProductCategory::find()->andWhere('is_deleted=:is AND product_category_parent_id IS NULL', [':is' => 0])->all();
        $slides = Slide::find()->andWhere('slide_is_active=:active', [':active' => 1])->orderBy(['slide_order' => SORT_ASC])->all();
        $articles = Article::find()->andWhere('is_deleted=:is AND article_status=:status', [':status' => Article::STATUS_PUBLISHED, ':is' => 0])->all();

        return $this->render('index', [
            'categories' => $categories,
            'slides' => $slides,
            'articles' => $articles
        ]);
    }

    public function actionDistribution()
    {
        BaseController::$page_caption = 'Distribution';

        $distributions = IndoDistribution::find()->all();
        $interDistributions = InternationalDistribution::find()->all();
        $segmentations = Segmentation::find()->all();

        $distCols = [];
        $index = 0;
        foreach($distributions as $i => $distribution) {
            $cityPerCol = ceil(count($distribution->indoDistributionDetails) / $distribution->dist_col);
            $distCols[$index]['cities'] = [];
            for($j = 0; $j < $distribution->dist_col; $j++) {
                $distCols[$index]['province_name'] = $j == 0 ? $distribution->dist_province_name : '';
                for($k = $cityPerCol * $j; $k < $cityPerCol * ($j+1); $k++) {
                    if (isset($distribution->indoDistributionDetails[$k])) {
                        $distCols[$index]['cities'][] = $distribution->indoDistributionDetails[$k]->dist_city_name;
                    }
                }
                $index++;
            }
        }

        return $this->render('distribution', [
            'distributions' => $distributions,
            'interDistributions' => $interDistributions,
            'segmentations' => $segmentations,
            'distCols' => $distCols
        ]);
    }

    public function actionContact()
    {
        BaseController::$page_caption = 'Contact Us';

        return $this->render('contact');
    }

    public function actionSearch()
    {
        BaseController::$page_caption = 'Contact Us';

        $articles = Article::find()->andFilterWhere(['like', 'article_title', $_GET['keyword']])->andWhere('is_deleted=:is', [':is' => 0])->all();
        $products = ProductCategory::find()->andFilterWhere(['like', 'product_category_name', $_GET['keyword']])->andWhere('is_deleted=:is', [':is' => 0])->all();
        $brands = ProductCategoryBrand::find()->andFilterWhere(['like', 'product_brand_name', $_GET['keyword']])->andWhere('is_deleted=:is', [':is' => 0])->all();

        $results = [];
        foreach($articles as $article) {
            $results[] = [
                'title' => $article->article_title,
                'url'   => Url::base(true).'/news/'.$article->article_slug,
                'desc'  => strip_tags(StringComponent::csubstr($article->article_short_description, 0, 100)),
            ];
        }

        foreach($products as $product) {
            $results[] = [
                'title' => $product->product_category_name,
                'url'   => Url::base(true).'/product/'.$product->product_category_slug,
                'desc'  => '',
            ];
        }

        foreach($brands as $brand) {
            $results[] = [
                'title' => $brand->product_brand_name,
                'url'   => Url::base(true).'/product/'.$brand->productCategory->product_category_slug,
                'desc'  => '',
            ];
        }

        return $this->render('search', [
            'results' => $results
        ]);
    }
}
