<?php

namespace frontend\models;

use yii\base\Model;
use yii\data\ActiveDataProvider;
use common\models\Product;

/**
 * ProductSearch represents the model behind the search form of `common\models\Product`.
 */
class ProductSearch extends Product
{
    public $keyword;

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['id', 'category_id', 'created_by', 'updated_by', 'is_deleted'], 'integer'],
            [['keyword', 'product_name', 'product_slug', 'product_desc', 'product_status', 'created_on', 'updated_on'], 'safe'],
            [['product_price', 'product_sale_price'], 'number'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function scenarios()
    {
        // bypass scenarios() implementation in the parent class
        return Model::scenarios();
    }

    /**
     * Creates data provider instance with search query applied
     *
     * @param array $params
     *
     * @return ActiveDataProvider
     */
    public function search($params)
    {
        $query = Product::find();
        $query->distinct()
            ->select([
                'product.id', 
                'product.product_name', 
                'product.product_price', 
                'product.product_sale_price', 
                'product.product_slug',
                'SUM(inventory.inventory_stock)'
            ])
            ->alias('product')
            ->joinWith(['inventories AS inventory'])
            ->groupBy('product.id')
            ->having('SUM(inventory.inventory_stock) > 0')
            ->orderBy(['COALESCE(product.updated_on, product.created_on)' => SORT_DESC]);

        // add conditions that should always apply here

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 9,
            ],
        ]);

        $this->load($params);

        if (!$this->validate()) {
            // uncomment the following line if you do not want to return any records when validation fails
            // $query->where('0=1');
            return $dataProvider;
        }

        // grid filtering conditions
        $query->andFilterWhere([
            'product.category_id' => $this->category_id,
            'product.product_status' => Product::STATUS_ACTIVE,
            'product.is_deleted' => '0',
        ]);

        if (!empty($this->keyword)) {
            $query->andFilterWhere(['like', 'product.product_name', $this->keyword]);
        }

        return $dataProvider;
    }

    public function searchNewArrival($params)
    {
        $query = Product::find();
        $query->distinct()
            ->select([
                'product.id', 
                'product.product_name', 
                'product.product_price', 
                'product.product_sale_price', 
                'product.product_slug',
                'SUM(inventory.inventory_stock)'
            ])
            ->alias('product')
            ->joinWith(['inventories AS inventory'])
            ->groupBy('product.id')
            ->having('SUM(inventory.inventory_stock) > 0')
            ->orderBy(['COALESCE(product.updated_on, product.created_on)' => SORT_DESC]);

        // add conditions that should always apply here

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 9,
            ],
        ]);

        $this->load($params);

        if (!$this->validate()) {
            // uncomment the following line if you do not want to return any records when validation fails
            // $query->where('0=1');
            return $dataProvider;
        }

        // grid filtering conditions
        $query->andFilterWhere([
            'product_status' => Product::STATUS_ACTIVE,
            'is_deleted' => '0',
        ]);

        return $dataProvider;
    }

    public function searchSale($params)
    {
        $query = Product::find();
        $query->distinct()
            ->select([
                'product.id', 
                'product.product_name', 
                'product.product_price', 
                'product.product_sale_price', 
                'product.product_slug',
                'SUM(inventory.inventory_stock)'
            ])
            ->alias('product')
            ->joinWith(['inventories AS inventory'])
            ->groupBy('product.id')
            ->having('SUM(inventory.inventory_stock) > 0')
            ->andWhere('product_sale_price IS NOT NULL')
            ->orderBy(['COALESCE(product.updated_on, product.created_on)' => SORT_DESC]);

        // add conditions that should always apply here

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 9,
            ],
        ]);

        $this->load($params);

        if (!$this->validate()) {
            // uncomment the following line if you do not want to return any records when validation fails
            // $query->where('0=1');
            return $dataProvider;
        }

        // grid filtering conditions
        $query->andFilterWhere([
            'product_status' => Product::STATUS_ACTIVE,
            'is_deleted' => '0',
        ]);

        return $dataProvider;
    }
}
